/*
 * Decompiled with CFR 0.152.
 */
package com.mongodb.internal.connection;

import com.mongodb.MongoException;
import com.mongodb.MongoInternalException;
import com.mongodb.MongoInterruptedException;
import com.mongodb.MongoServerUnavailableException;
import com.mongodb.MongoTimeoutException;
import com.mongodb.annotations.ThreadSafe;
import com.mongodb.assertions.Assertions;
import com.mongodb.internal.Locks;
import com.mongodb.internal.connection.Pool;
import com.mongodb.internal.thread.InterruptionUtil;
import com.mongodb.lang.Nullable;
import java.util.Deque;
import java.util.Iterator;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Consumer;
import java.util.function.Supplier;

public class ConcurrentPool<T>
implements Pool<T> {
    public static final int INFINITE_SIZE = Integer.MAX_VALUE;
    private final int maxSize;
    private final ItemFactory<T> itemFactory;
    private final Deque<T> available = new ConcurrentLinkedDeque<T>();
    private final StateAndPermits stateAndPermits;
    private final String poolClosedMessage;

    public ConcurrentPool(int maxSize, ItemFactory<T> itemFactory) {
        this(maxSize, itemFactory, "The pool is closed");
    }

    public ConcurrentPool(int maxSize, ItemFactory<T> itemFactory, String poolClosedMessage) {
        Assertions.assertTrue(maxSize > 0);
        this.maxSize = maxSize;
        this.itemFactory = itemFactory;
        this.stateAndPermits = new StateAndPermits(maxSize, this::poolClosedException);
        this.poolClosedMessage = Assertions.notNull("poolClosedMessage", poolClosedMessage);
    }

    @Override
    public void release(T t) {
        this.release(t, false);
    }

    @Override
    public void release(T t, boolean prune) {
        if (t == null) {
            throw new IllegalArgumentException("Can not return a null item to the pool");
        }
        if (this.stateAndPermits.closed()) {
            this.close(t);
            return;
        }
        if (prune) {
            this.close(t);
        } else {
            this.available.addLast(t);
        }
        this.stateAndPermits.releasePermit();
    }

    @Override
    public T get() {
        return this.get(-1L, TimeUnit.MILLISECONDS);
    }

    @Override
    public T get(long timeout, TimeUnit timeUnit) {
        if (!this.stateAndPermits.acquirePermit(timeout, timeUnit)) {
            throw new MongoTimeoutException(String.format("Timeout waiting for a pooled item after %d %s", new Object[]{timeout, timeUnit}));
        }
        T t = this.available.pollLast();
        if (t == null) {
            t = this.createNewAndReleasePermitIfFailure();
        }
        return t;
    }

    @Nullable
    T getImmediate() {
        T element = null;
        if (this.stateAndPermits.acquirePermitImmediate() && (element = (T)this.available.pollLast()) == null) {
            this.stateAndPermits.releasePermit();
        }
        return element;
    }

    public void prune() {
        int maxIterations = this.available.size();
        int numIterations = 0;
        for (T cur : this.available) {
            if (this.itemFactory.shouldPrune(cur) && this.available.remove(cur)) {
                this.close(cur);
            }
            if (++numIterations != maxIterations) continue;
            break;
        }
    }

    public void ensureMinSize(int minSize, Consumer<T> initAndRelease) {
        while (this.getCount() < minSize && this.stateAndPermits.acquirePermit(0L, TimeUnit.MILLISECONDS)) {
            initAndRelease.accept(this.createNewAndReleasePermitIfFailure());
        }
    }

    private T createNewAndReleasePermitIfFailure() {
        try {
            T newMember = this.itemFactory.create();
            if (newMember == null) {
                throw new MongoInternalException("The factory for the pool created a null item");
            }
            return newMember;
        }
        catch (Exception e) {
            this.stateAndPermits.releasePermit();
            throw e;
        }
    }

    boolean acquirePermit(long timeout, TimeUnit timeUnit) {
        return this.stateAndPermits.acquirePermit(timeout, timeUnit);
    }

    @Override
    public void close() {
        if (this.stateAndPermits.close()) {
            Iterator<T> iter = this.available.iterator();
            while (iter.hasNext()) {
                T t = iter.next();
                this.close(t);
                iter.remove();
            }
        }
    }

    int getMaxSize() {
        return this.maxSize;
    }

    public int getInUseCount() {
        return this.maxSize - this.stateAndPermits.permits();
    }

    public int getAvailableCount() {
        return this.available.size();
    }

    public int getCount() {
        return this.getInUseCount() + this.getAvailableCount();
    }

    public String toString() {
        return "pool:  maxSize: " + ConcurrentPool.sizeToString(this.maxSize) + " availableCount " + this.getAvailableCount() + " inUseCount " + this.getInUseCount();
    }

    private void close(T t) {
        try {
            this.itemFactory.close(t);
        }
        catch (Exception exception) {
            // empty catch block
        }
    }

    void ready() {
        this.stateAndPermits.ready();
    }

    void pause(Supplier<MongoException> causeSupplier) {
        this.stateAndPermits.pause(causeSupplier);
    }

    MongoServerUnavailableException poolClosedException() {
        return new MongoServerUnavailableException(this.poolClosedMessage);
    }

    static boolean isPoolClosedException(Throwable e) {
        return e instanceof MongoServerUnavailableException;
    }

    static String sizeToString(int size) {
        return size == Integer.MAX_VALUE ? "infinite" : Integer.toString(size);
    }

    public static interface ItemFactory<T> {
        public T create();

        public void close(T var1);

        public boolean shouldPrune(T var1);
    }

    @ThreadSafe
    private static final class StateAndPermits {
        private final Supplier<MongoServerUnavailableException> poolClosedExceptionSupplier;
        private final ReentrantLock lock;
        private final Condition permitAvailableOrClosedOrPausedCondition;
        private volatile boolean paused;
        private volatile boolean closed;
        private final int maxPermits;
        private volatile int permits;
        @Nullable
        private Supplier<MongoException> causeSupplier;

        StateAndPermits(int maxPermits, Supplier<MongoServerUnavailableException> poolClosedExceptionSupplier) {
            this.poolClosedExceptionSupplier = poolClosedExceptionSupplier;
            this.lock = new ReentrantLock();
            this.permitAvailableOrClosedOrPausedCondition = this.lock.newCondition();
            this.paused = false;
            this.closed = false;
            this.maxPermits = maxPermits;
            this.permits = maxPermits;
            this.causeSupplier = null;
        }

        int permits() {
            return this.permits;
        }

        boolean acquirePermitImmediate() {
            return Locks.withLock((Lock)this.lock, () -> {
                this.throwIfClosedOrPaused();
                if (this.permits > 0) {
                    --this.permits;
                    return true;
                }
                return false;
            });
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         * Enabled aggressive block sorting
         * Enabled unnecessary exception pruning
         * Enabled aggressive exception aggregation
         */
        boolean acquirePermit(long timeout, TimeUnit unit) throws MongoInterruptedException {
            long remainingNanos = unit.toNanos(timeout);
            Locks.lockInterruptibly(this.lock);
            try {
                while (this.permits == 0 & !this.throwIfClosedOrPaused()) {
                    try {
                        if (timeout < 0L || remainingNanos == Long.MAX_VALUE) {
                            this.permitAvailableOrClosedOrPausedCondition.await();
                            continue;
                        }
                        if (remainingNanos < 0L) {
                            boolean bl = false;
                            return bl;
                        }
                        remainingNanos = this.permitAvailableOrClosedOrPausedCondition.awaitNanos(remainingNanos);
                    }
                    catch (InterruptedException e) {
                        throw InterruptionUtil.interruptAndCreateMongoInterruptedException(null, e);
                    }
                }
                Assertions.assertTrue(this.permits > 0);
                --this.permits;
                boolean bl = true;
                return bl;
            }
            finally {
                this.lock.unlock();
            }
        }

        void releasePermit() {
            Locks.withLock((Lock)this.lock, () -> {
                Assertions.assertTrue(this.permits < this.maxPermits);
                ++this.permits;
                this.permitAvailableOrClosedOrPausedCondition.signal();
            });
        }

        void pause(Supplier<MongoException> causeSupplier) {
            Locks.withLock((Lock)this.lock, () -> {
                if (!this.paused) {
                    this.paused = true;
                    this.permitAvailableOrClosedOrPausedCondition.signalAll();
                }
                this.causeSupplier = Assertions.assertNotNull(causeSupplier);
            });
        }

        void ready() {
            if (this.paused) {
                Locks.withLock((Lock)this.lock, () -> {
                    this.paused = false;
                    this.causeSupplier = null;
                });
            }
        }

        boolean close() {
            if (!this.closed) {
                return Locks.withLock((Lock)this.lock, () -> {
                    if (!this.closed) {
                        this.closed = true;
                        this.permitAvailableOrClosedOrPausedCondition.signalAll();
                        return true;
                    }
                    return false;
                });
            }
            return false;
        }

        boolean throwIfClosedOrPaused() {
            if (this.closed) {
                throw this.poolClosedExceptionSupplier.get();
            }
            if (this.paused) {
                throw Assertions.assertNotNull(Assertions.assertNotNull(this.causeSupplier).get());
            }
            return false;
        }

        boolean closed() {
            return this.closed;
        }
    }
}

