/**
 * i-net software provides programming examples for illustration only, without warranty
 * either expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and/or fitness for a particular purpose. This programming example
 * assumes that you are familiar with the programming language being demonstrated and
 * the tools used to create and debug procedures. i-net software support professionals
 * can help explain the functionality of a particular procedure, but they will not modify
 * these examples to provide added functionality or construct procedures to meet your
 * specific needs.
 *
 * Copyright © 1999-2025 i-net software GmbH, Berlin, Germany.
**/
package com.inet.taskplanner.openweathermap;

import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.inet.config.structure.model.LocalizedKey;
import com.inet.id.GUID;
import com.inet.taskplanner.server.api.common.SummaryEntry;
import com.inet.taskplanner.server.api.error.ValidationException;
import com.inet.taskplanner.server.api.field.Field;
import com.inet.taskplanner.server.api.field.LinkField;
import com.inet.taskplanner.server.api.field.SelectField;
import com.inet.taskplanner.server.api.field.TextField;
import com.inet.taskplanner.server.api.job.ConditionDefinition;
import com.inet.taskplanner.server.api.job.ConditionInfo;
import com.inet.taskplanner.server.api.job.JobDefinition;
import com.inet.taskplanner.server.api.job.JobFactory;
import com.inet.taskplanner.server.api.job.JobInfo;
import com.inet.taskplanner.server.api.job.JobSummaryInfo;
import com.inet.taskplanner.server.api.result.ResultFlavor;

/**
 * The factory for an open weather map job that queries the current weather information.<br>
 * In this implementation the job is defined and validated.
 */
public class OpenWeatherMapJobFactory extends JobFactory<OpenWeatherMapJob> {

    static final String PROPERTY_LOCATION    = "location";
    static final String PROPERTY_APIKEY      = "apikey";
    static final String PROPERTY_LINK        = "link";
    static final String PROPERTY_UNITS       = "units";
    static final String PROPERTY_TEMPERATURE = "temperature";

    /**
     * Creates the factory for the weather job
     */
    public OpenWeatherMapJobFactory() {
        // Provide a unique ID for this factory
        super( "job.openweathermap" );
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public java.util.List<ResultFlavor> getResultFlavors( com.inet.taskplanner.server.api.job.JobDefinition definition ) {
        // A text result will be returned
        return Arrays.asList( ResultFlavor.TEXT );
    };

    /**
     * {@inheritDoc}
     */
    @Override
    public JobInfo getInformation( @Nullable GUID taskID ) {
        // Define an icon for the job. It must be 32x32 pixels. The icon itself should be white with transparent background.
        URL iconURL = getClass().getResource( "/com/inet/taskplanner/openweathermap/taskplanner_openweather_32.png" );

        // All settings that can be configurated must be defined as field and added to the list of fields.
        List<Field> fields = new ArrayList<Field>();

        // The api key muss be entered
        fields.add( new TextField( PROPERTY_APIKEY, TaskPlannerOpenWeatherMapServerPlugin.MSG.getMsg( "taskplanner.openweathermap.apikey" ) ) );

        // Shows a link to the given targetURL. The "" as second parameter hides the label in front of the link.
        fields.add( new LinkField( PROPERTY_LINK, "", TaskPlannerOpenWeatherMapServerPlugin.MSG.getMsg( "taskplanner.openweathermap.link" ), TaskPlannerOpenWeatherMapServerPlugin.MSG.getMsg( "taskplanner.openweathermap.link.text" ) ) );

        // Adds a text field for the location and sets a placeholder with information about the data to be entered
        TextField location = new TextField( PROPERTY_LOCATION, TaskPlannerOpenWeatherMapServerPlugin.MSG.getMsg( "taskplanner.openweathermap.location" ) );
        location.setPlaceholder( TaskPlannerOpenWeatherMapServerPlugin.MSG.getMsg( "taskplanner.openweathermap.location.placeholder" ) );
        fields.add( location );

        // Gets all available units and adds them as a select field.
        ArrayList<LocalizedKey> values = new ArrayList<LocalizedKey>();
        for( Units unit : Units.values() ) {
            values.add( new LocalizedKey( unit.name(), TaskPlannerOpenWeatherMapServerPlugin.MSG.getMsg( "taskplanner.openweathermap.units." + unit.name() ) ) );
        }
        SelectField units = new SelectField( PROPERTY_UNITS, TaskPlannerOpenWeatherMapServerPlugin.MSG.getMsg( "taskplanner.openweathermap.units" ), values );
        units.setValue( Units.DEFAULT.name() ); // Sets the default value
        fields.add( units );

        // This job factory provides the following list of placeholder keys that will be available in result action as e.g. [weather.city].
        List<String> placeholderKeys = new ArrayList<String>();
        placeholderKeys.add( "weather.city" );
        placeholderKeys.add( "weather.country" );
        placeholderKeys.add( "weather.description" );
        placeholderKeys.add( "weather.temperature" );
        placeholderKeys.add( "weather.pressure" );
        placeholderKeys.add( "weather.humidity" );
        placeholderKeys.add( "weather.windspeed" );
        placeholderKeys.add( "weather.sunrise" );
        placeholderKeys.add( "weather.sunset" );

        // In addition to the previous field, the factory supports a condition that only enables result processing if the condition is fulfilled.
        // Here the temperature must be above an enered value.
        List<Field> conditionFields = new ArrayList<Field>();
        TextField temperature = new TextField( PROPERTY_TEMPERATURE, TaskPlannerOpenWeatherMapServerPlugin.MSG.getMsg( "taskplanner.openweathermap.temperature" ) );
        conditionFields.add( temperature );
        ConditionInfo conditionInfo = new ConditionInfo( conditionFields );

        // Creates the job information with unique ID, name, description, iconURL, NO helpkey, condition, fields and placeholder keys.
        JobInfo info = new JobInfo( getExtensionName(), TaskPlannerOpenWeatherMapServerPlugin.MSG.getMsg( "taskplanner.openweathermap.name" ), TaskPlannerOpenWeatherMapServerPlugin.MSG.getMsg( "taskplanner.openweathermap.description" ), iconURL, null, conditionInfo, fields, placeholderKeys );
        return info;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected void validate( @Nonnull JobDefinition definition, @Nullable GUID taskID ) throws ValidationException {
        // Validate: the api key must be set
        String apikey = definition.getProperty( PROPERTY_APIKEY );
        if( apikey == null || apikey.trim().isEmpty() ) {
            throw new ValidationException( TaskPlannerOpenWeatherMapServerPlugin.MSG.getMsg( "taskplanner.openweathermap.validation.apikey.empty" ) );
        }
        // Validate the locatoin must be set
        String location = definition.getProperty( PROPERTY_LOCATION );
        if( location == null || location.trim().isEmpty() ) {
            throw new ValidationException( TaskPlannerOpenWeatherMapServerPlugin.MSG.getMsg( "taskplanner.openweathermap.validation.location.empty" ) );
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected void validateCondition( @Nonnull JobDefinition definition ) throws ValidationException {
        // Validate: The temperature must be a valid number
        ConditionDefinition condition = definition.getCondition();
        if ( condition != null ) {  
            String temperature = condition.getProperty( PROPERTY_TEMPERATURE );
            try {
                Double.parseDouble( temperature );
            } catch( Throwable t ) {
                throw new ValidationException( TaskPlannerOpenWeatherMapServerPlugin.MSG.getMsg( "taskplanner.openweathermap.validation.temperature.invalid" ) );
            }
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected OpenWeatherMapJob createInstanceFrom( @Nonnull JobDefinition definition, @Nullable GUID taskID ) {
        // Create the instance for the api key, location and units setting
        return new OpenWeatherMapJob( definition.getProperty( PROPERTY_APIKEY ), definition.getProperty( PROPERTY_LOCATION ), definition.getProperty( PROPERTY_UNITS ), definition.getCondition() );
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public JobSummaryInfo getSummary( @Nonnull JobDefinition definition ) {
        // Build a summary and add the location
        List<SummaryEntry> info = new ArrayList<SummaryEntry>();
        info.add( new SummaryEntry( TaskPlannerOpenWeatherMapServerPlugin.MSG.getMsg( "taskplanner.openweathermap.location" ), definition.getProperty( PROPERTY_LOCATION ) ) );

        List<SummaryEntry> conditionInfo = null;

        // If a condition is set, show the configured settings
        ConditionDefinition condition = definition.getCondition();
        if( condition != null ) {
            conditionInfo = new ArrayList<SummaryEntry>();
            String unitValue = definition.getProperty( PROPERTY_UNITS );
            Units units = Units.DEFAULT;
            if( unitValue != null ) {
                try {
                    units = Units.valueOf( unitValue );
                } catch( IllegalArgumentException iae ) {
                    // Fallback to DEFAULT
                }
            }
            conditionInfo.add( new SummaryEntry( TaskPlannerOpenWeatherMapServerPlugin.MSG.getMsg( "taskplanner.openweathermap.temperature" ), condition.getProperty( PROPERTY_TEMPERATURE ) + " " + units.getTemperatureUnit() ) );
        }

        return new JobSummaryInfo( info, conditionInfo );
    }

}
