/**
 * i-net software provides programming examples for illustration only, without warranty
 * either expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and/or fitness for a particular purpose. This programming example
 * assumes that you are familiar with the programming language being demonstrated and
 * the tools used to create and debug procedures. i-net software support professionals
 * can help explain the functionality of a particular procedure, but they will not modify
 * these examples to provide added functionality or construct procedures to meet your
 * specific needs.
 *
 * Copyright © 1999-2026 i-net software GmbH, Berlin, Germany.
**/
package com.inet.authentication.oauth.ex;

import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Set;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.inet.authentication.AuthenticationDescription;
import com.inet.config.structure.model.ConfigCategory;
import com.inet.config.structure.model.ConfigCondition;
import com.inet.config.structure.model.ConfigConditionAction;
import com.inet.config.structure.model.ConfigGroup;
import com.inet.config.structure.model.ConfigProperty;
import com.inet.config.structure.model.ConfigPropertyGroup;
import com.inet.config.structure.model.ConfigValidationMsg;
import com.inet.config.structure.provider.AbstractStructureProvider;
import com.inet.config.structure.provider.ConditionGenerator;
import com.inet.config.structure.provider.ConfigStructureSettings;
import com.inet.config.structure.provider.ConfigValidator;

/**
 * Configuration of the OAuth sample service. This extends the UI with the needed extra values.
 * ClientID and ClientSecret will already be shown without this StructureProvider.
 * This is only needed if you do not hard code the extra values of the provider.
 */
public class OAuthExStructureProvider extends AbstractStructureProvider {

    private static final String GROUP_OAUTH = "oauth.action";

    public static final OAuthExStructureProvider INSTANCE = new OAuthExStructureProvider();

    /**
     * {@inheritDoc}
     */
    @Override
    public void addConditionsTo( @Nonnull List<ConfigConditionAction> actions, @Nonnull String categoryKey, boolean advanced, @Nonnull ConfigStructureSettings settings ) {
        switch( categoryKey ) {
            case GROUP_OAUTH:
                // show only if this sample OAuth provider is selected
                ConditionGenerator g = new ConditionGenerator();
                ConfigCondition condition = g.createCondition().property( AuthenticationDescription.SYSTEM_LOGINTYPE ).equals().value( OAuthExServerDescription.NAME );
                actions.add( g.visibleActionFor( condition, OAuthExServerDescription.OAUTH_AUTHENTICATION_URL ) );
                actions.add( g.visibleActionFor( condition, OAuthExServerDescription.OAUTH_TOKEN_URL ) );
                actions.add( g.visibleActionFor( condition, OAuthExServerDescription.OAUTH_USERINFO_URL ) );
                break;
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void addPropertiesTo( @Nonnull Set<ConfigProperty> configProperties, @Nonnull String propertyGroupKey, boolean advanced, @Nonnull ConfigStructureSettings settings ) {
        switch( propertyGroupKey ) {
            case GROUP_OAUTH:
                // sortid should be > 100 to show it after the OAuth provider selection
                // sortid should be < 400 to show it before the client ID.
                configProperties.add( new ConfigProperty(150, OAuthExServerDescription.OAUTH_AUTHENTICATION_URL, ConfigProperty.SIMPLE_TEXT, "Authentication URL", "", null, null ) );
                configProperties.add( new ConfigProperty(160, OAuthExServerDescription.OAUTH_TOKEN_URL, ConfigProperty.SIMPLE_TEXT, "Token URL", "", null, null ) );
                configProperties.add( new ConfigProperty(160, OAuthExServerDescription.OAUTH_USERINFO_URL, ConfigProperty.SIMPLE_TEXT, "Userinfo URL", "", null, null ) );
                break;
            default:
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void validate( @Nullable String categoryKey, @Nonnull ArrayList<ConfigValidationMsg> messages, @Nonnull ArrayList<ConfigConditionAction> conditions, @Nonnull ConfigStructureSettings settings ) {
        if( GROUP_OAUTH.equals( categoryKey ) ) {
            String loginType = settings.getValue( AuthenticationDescription.SYSTEM_LOGINTYPE );
            if( Objects.equals( loginType, OAuthExServerDescription.NAME ) ) {
                // prevent wrong settings if possible, add all desired checks here
                ConfigValidator validator = new ConfigValidator( messages, settings );
                validator.checkNotEmpty( OAuthExServerDescription.OAUTH_AUTHENTICATION_URL );
                validator.checkNotEmpty( OAuthExServerDescription.OAUTH_TOKEN_URL );
                validator.checkNotEmpty( OAuthExServerDescription.OAUTH_USERINFO_URL );
            }
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void addGroupsTo( @Nonnull Set<ConfigGroup> groups, boolean advanced, @Nonnull ConfigStructureSettings settings ) {
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void addCategoriesTo( @Nonnull Set<ConfigCategory> categories, @Nonnull String groupKey, boolean advanced, @Nonnull ConfigStructureSettings settings ) {
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public URL getCategoryIcon( @Nonnull String categoryKey ) {
        return null;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void addPropertyGroupsTo( @Nonnull Set<ConfigPropertyGroup> configPropertyGroups, @Nonnull String categoryKey, boolean advanced, @Nonnull ConfigStructureSettings settings ) {
    }

}
