/**
 * i-net software provides programming examples for illustration only, without warranty
 * either expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and/or fitness for a particular purpose. This programming example
 * assumes that you are familiar with the programming language being demonstrated and
 * the tools used to create and debug procedures. i-net software support professionals
 * can help explain the functionality of a particular procedure, but they will not modify
 * these examples to provide added functionality or construct procedures to meet your
 * specific needs.
 *
 * Copyright © 1999-2025 i-net software GmbH, Berlin, Germany.
**/
package com.inet.authentication.passwordfile;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.inet.authentication.AuthenticationDescription;
import com.inet.authentication.AuthenticationDescription.BasicSupport;
import com.inet.authentication.AuthenticationProvider;
import com.inet.authentication.LoginProcessor;
import com.inet.authentication.passwordfile.structure.PasswordFileAuthStructureProvider;
import com.inet.config.Configuration;
import com.inet.http.servlet.ClientLocale;

import jakarta.servlet.http.HttpServletRequest;

/**
 * AuthenticationProvider for password file.
 */
public class PasswordFileAuthProvider implements AuthenticationProvider {

    // the login type saved in the configuration, every AuthenticationProvider must return an unique name
    public static final String NAME         = "passwordfile";

    // cache for the loaded password file
    private final PasswordFile passwordFile = new PasswordFile();

    /**
     * {@inheritDoc}
     */
    @Override
    public @Nonnull String name() {
        return NAME;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public int getPriority() {
        return 444;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public @Nonnull LoginProcessor create( AuthenticationDescription description ) {
        String path = description.getSettings().get( PasswordFileAuthStructureProvider.PASSWORD_FILE );
        passwordFile.setPasswordFile( path );
        return new PasswordFileAuthWUI( description, passwordFile );
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public @Nonnull String getDisplayName( @Nullable Map<String, String> settings ) {
        try {
            ResourceBundle bundle = ResourceBundle.getBundle( "com.inet.authentication.passwordfile.structure.i18n.ConfigStructure", ClientLocale.getThreadLocale() );
            return bundle.getString( name() );
        } catch( MissingResourceException ex ) {
            return name();
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public AuthenticationDescription getAuthenticationDescription( @Nonnull Map<String, String> settings, boolean isApplication, boolean isProxy ) {
        return new AuthenticationDescription( this, settings, name(), BasicSupport.No );
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public LoginProcessor createLoginProcessor( AuthenticationDescription description, @Nullable String authorization, HttpServletRequest request, boolean isApplication ) {
        if( authorization == null || !authorization.startsWith( "Digest " ) ) {
            return null;
        }
        return create( description );
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public @Nonnull Map<String, String> applySettings( Map<String, String> clientProperties, Configuration config, @Nonnull List<Map<String, String>> originalInConfig ) {
        HashMap<String, String> result = new HashMap<>();
        result.put( PasswordFileAuthStructureProvider.PASSWORD_FILE, clientProperties.get( PasswordFileAuthStructureProvider.PASSWORD_FILE ) );
        return result;
    }
}
