/**
 * Kytos base util functions
 */
export class Kytos {

	/**
	 * Creates a DOM node and assigns the optionally given classes
	 * @param tagName the tag name of the node to be created
	 * @param classes otional array of class names
	 * @return the created node  
	 */	
	static createNode(tagName, ...classes) {
		if (tagName == undefined || tagName.length==0) {
			throw new Error('A tagName must be specified');
		}
		let node = document.createElement(tagName);
		if (classes != undefined && classes.length > 0) {
			node.classList.add(...classes);
		}
		return node;
	};
	
	/**
	 * Creates a DOM node, assigns the optionally given classes and sets the text content, if provided
	 * @param tagName the tag name of the node to be created
	 * @param textContent optional text content to be set
	 * @param classes otional array of class names
	 * @return the created node  
	 */	
	static createNodeWithText(tagName, textContent, ...classes) {
		let node = Kytos.createNode(tagName, ...classes);
		if (textContent != undefined && textContent.length > 0) {
			node.textContent = textContent; 
		}
		return node;
	};

    /**
     * Registers a mutation observer on the parentNode in order to call the callback function when the node is removed.
     * @param node the node that is watched and notified when removed
     * @param callback function that is called when the node was removed. Provides the removed node as only argument
     * @param parentNode optional node to register the MutationObserver at. If not provided, the parentNode of the observed node will be used
     * @return the MutationObserver instance to e.g. allow to disconnect it  
     */
    static onNodeRemoved(node, callback, parentNode) {
        let observer = new MutationObserver((mutations) => {
            mutations.forEach((mutation) => {
                if (mutation.type === "childList") {
                    for (let removedNode of mutation.removedNodes) {
                        if (removedNode == node) {
                            callback(node);
                        } else if (mutation.removedNodes[0].compareDocumentPosition(node) & Node.DOCUMENT_POSITION_CONTAINED_BY) { // parent including "node" was removed
                            callback(node);
                        }
                    }
                }
            });
        });
        if (parentNode != undefined) {
            observer.observe(parentNode, {
                childList: true,
                subtree: true
            });
        } else {
            observer.observe(node.parentNode, {
                childList: true
            });
        }
        return observer;
    };
    
    /**
     * Returns the value of a key in the given model. Will be called recursivly if the key still contains a dot.
     * @param propName the name of the property
     * @param model the model to get the property value from
     * @return the value of the given property name in the model
     */
    static getModelValue(propName, model) {
        if (model == undefined) {
            return undefined;
        }
        let index = propName.indexOf('.');
        let subProps = undefined; 
        if (index>0) { 
            subProps = propName.substring(index+1);
            propName = propName.substring(0,index);
        }
        let value = model[propName];
        if (subProps != undefined) {
            return Kytos.getModelValue(subProps, value);
        }
        return value;
    };
    
	/**
	 * Decrypts and converts the given data into a JSON object
	 * @param data the data to be decrypted and converted
	 * @return the resulting JSON data
	 */
	static extractJsonBytes(data) {
	    if (data && data instanceof ArrayBuffer) {
	        var buffer = new Uint8Array(data);  
	        var qwirks = Kytos.utf8ArrayToStr(buffer);
	        if (qwirks.length>0) {
	            var decrypted = Kytos.decrypt(qwirks);
	            try {
	                if (decrypted == undefined) {
	                    return undefined;
	                }
	                return JSON.parse(decrypted);
	            } catch (error) {
	                throw error + ": " + Kytos.html2text(qwirks);
	            }
	        }
	    } else {
	        return data;
	    }
	};
	
	/**
	 * Returns the json object for the given received data. It will be decrypted and converted if the data is a string. For IE8, the stream is not encrypted, so the object is the resulting instance.
	 * @param data the received data from POST request. Needs to be deserialized if it is type of "string"
	 * @returns the deserialized json object
	 */
	static extractJson(data) {
	    if (data && typeof data == "string") {
	        var decrypted = Kytos.decrypt(data);
	        try {
	            if (decrypted == undefined) {
	                return undefined;
	            }
	            return JSON.parse(decrypted);
	        } catch (error) {
	            throw error + ": " + Kytos.html2text(data);
	        }
	    } else {
	        return data;
	    }
	};
	
	/**
	 * decrypt a json from the server
	 * @param qwirks the string to decrypt
	 * @returns the decrypted json
	 */
	static decrypt(qwirks) {
	    if (qwirks && qwirks.charCodeAt(0) === 123 /* '{' */) {
	        return qwirks;
	    }
	    if (qwirks && qwirks == "null") {
	        return undefined;
	    }
	    var l = qwirks.length;
	    var result = [];
	    var xor = 0x47;
	    for (let i = 0; i < l; i++) {
	        var code = qwirks.charCodeAt(i);
	        xor = (code ^ xor);
	        result.push(String.fromCharCode(xor));
	        xor &= 0xFF;
	    }
	    return result.join("");
	};
	
	/**
	 * encrypt a json to save transfer to the server
	 * @param json the json to encrypt
	 * @returns the encrypted json
	 */
	static encrypt = function (json) {
	    var l = json.length;
	    var result = [];
	    var xor = 0x47;
	    for (let i = 0; i < l; i++) {
	        var code = json.charCodeAt(i);
	        var targetCharCode = code ^ xor;
	        var targetChar = String.fromCharCode(targetCharCode);
	        result.push(targetChar);
	        xor = (code & 0xFF);
	    }
	    return result.join("");
	};
	
	/**
	 * Converts the given html content to plain text, matchng the line breaks and so on
	 * @param source the html content to be converted
	 * @param htmlLight whether html light content should be returned for e.g. htmleditor hugeRTE
	 * @return the converted content
	 */
	static html2text(source, htmlLight) {
        if (source == undefined || source == null) {
            return "";
        }
        var result = source;

        result = result.replace(/\r/gi, " ");
        result = result.replace(/\n/gi, " ");
        result = result.replace(/\t/gi, "");
        result = result.replace(/\s{2,}/gi, " ");
        result = result.replace(/<\s*head(\s.*?)*>.*?<\/\s*head(\s.*?)*>/gi, "");
        result = result.replace(/<\s*script.*?>.*?<\/\s*script.*?>/gi, "");
        result = result.replace(/<\s*script.*?>/gi, "");
        result = result.replace(/<\s*style.*?>.*?<\/\s*style.*?>/gi, "");
        result = result.replace(/<\s*link.*?>/gi, "");
        result = result.replace(/<\s*html.*?>/gi, "");
        result = result.replace(/<\s*body.*?>/gi, "");
        result = result.replace(/<!--.*?-->/gi, "");

        // Absätze und Umbrüche
        result = result.replace(/<\s*p.*?>/gi, "\n\n");
        result = result.replace(/<\s*br.*?>/gi, "\n");
        result = result.replace(/<\s*div.*?>/gi, "\n");
        result = result.replace(/<\s*h\d.*?>/gi, "\n");
        result = result.replace(/<\s*\/\s*h\d.*?>/gi, "\n");

        //Tabelle
        result = result.replace(/<\s*tr.*?>/gi, "");
        result = result.replace(/<\s*\/\s*tr.*?>/gi, "\n");
        result = result.replace(/<\s*td.*?>/gi, "");
        result = result.replace(/<\s*\/\s*td.*?>/gi, "\t");
        result = result.replace(/<\s*th.*?>/gi, "");
        result = result.replace(/<\s*\/\s*th.*?>/gi, "\t");

        // Listen
        result = result.replace(/<\s*ul.*?>/gi, "\n");
        result = result.replace(/<\s*\/\s*ul.*?>/gi, "\n");
        result = result.replace(/<\s*li.*?>/gi, "* ");
        result = result.replace(/<\s*\/\s*li.*?>/gi, "\n");

        // restliche Tags strippen
        result = result.replace(/<[^>]*>/gi, "");

        // Durch ein <p> oder <br> am Anfang entstehen 1 bis 2 \n, die wir netterweise wegkürzen
        if (htmlLight) {
            result = result.trim();
            
            result = result.replace(/\n\n/gi, "<p class=\"hddefault\">");
            result = result.replace(/\n/gi, "<br>");
            result = result.replace(/\t/gi, "&nbsp;&nbsp;&nbsp;&nbsp;");
        } else {
            if (result.indexOf("\n\n") == 0) {
                result = result.substring(2);
            }
            result = result.replace(/\n\n/gi, "<p class=\"hddefault\">");
            result = result.replace(/<p>/gi, "\n");
        }
        return result;
    };
	
	/**
	 * Converts the given utf8 array into a string
	 * @param the array to be converted
	 * @return the string result
	 */
	static utf8ArrayToStr(array) {
	    let charCache = new Array(128);  // Preallocate the cache for the common single byte chars
	    let charFromCodePt = String.fromCodePoint || String.fromCharCode;
	    let result = [];
        let codePt, byte1;
        let buffLen = array.length;

        result.length = 0;

        for (let i = 0; i < buffLen;) {
            byte1 = array[i++];

            if (byte1 <= 0x7F) {
                codePt = byte1;
            } else if (byte1 <= 0xDF) {
                codePt = ((byte1 & 0x1F) << 6) | (array[i++] & 0x3F);
            } else if (byte1 <= 0xEF) {
                codePt = ((byte1 & 0x0F) << 12) | ((array[i++] & 0x3F) << 6) | (array[i++] & 0x3F);
            } else {
                codePt = ((byte1 & 0x07) << 18) | ((array[i++] & 0x3F) << 12) | ((array[i++] & 0x3F) << 6) | (array[i++] & 0x3F);
                if (!String.fromCodePoint) {
                    //Converting to UTF-16
                    codePt -= 0x10000;
                    var surrogate = 0xD800 | (codePt >> 10 ); // high Surrogate
                    result.push(charCache[surrogate] || (charCache[surrogate] = charFromCodePt(surrogate)));
                    codePt = 0xDC00 | ( codePt & 0x3FF ); // low Surrogate
                }
            }

            result.push(charCache[codePt] || (charCache[codePt] = charFromCodePt(codePt)));
        }

        return result.join('');
    };
	
	/**
	 * Converts the given string into an utf8 array
	 * @param atr the stirng to be converted
	 * @return the resulting array
	 */
	static toUTF8Array(str) {
	    var utf8 = [];
	    for (let i=0; i < str.length; i++) {
	        var charcode = str.charCodeAt(i);
	        if (charcode < 0x80) {
	            utf8.push(charcode);
	        } else if (charcode < 0x800) {
	            utf8.push(0xc0 | (charcode >> 6), 
	                      0x80 | (charcode & 0x3f));
	        } else if (charcode < 0xd800 || charcode >= 0xe000) {
	            utf8.push(0xe0 | (charcode >> 12), 
	                      0x80 | ((charcode>>6) & 0x3f), 
	                      0x80 | (charcode & 0x3f));
	        } else {
	            // surrogate pair
	            i++;
	            // UTF-16 encodes 0x10000-0x10FFFF by
	            // subtracting 0x10000 and splitting the
	            // 20 bits of 0x0-0xFFFFF into two halves
	            charcode = 0x10000 + (((charcode & 0x3ff)<<10)
	                      | (str.charCodeAt(i) & 0x3ff));
	            utf8.push(0xf0 | (charcode >>18), 
	                      0x80 | ((charcode>>12) & 0x3f), 
	                      0x80 | ((charcode>>6) & 0x3f), 
	                      0x80 | (charcode & 0x3f));
	        }
	    }
	    return utf8;
	};
}
