/**
 * Event emitter class for sending an retrieving events over different components.
 * 
 * Usage:
 *     let unregisterFunction = EventEmitter.on('custom.event', (data) => { console.log(data); });
 *     EventEmitter.fire('custom.event', {label:'event test'});
 *     EventEmitter.off('custom.event', listener); or unregisterFunction(); // must be the same listener as "on"
 */
export class EventEmitter {

	static #eventListener = {};	

	/*****************************************************************************************************************************************************************************************************************************/
	/*****************************************************************************************************************************************************************************************************************************/
	/** API methods and members                                                                                                                                                                                                 **/
	/*****************************************************************************************************************************************************************************************************************************/
	/*****************************************************************************************************************************************************************************************************************************/

	/**
	 * Fires the event with the given name and optional data
	 * @param eventName the name of the event
	 * @param data the optional data to be send
	 */
	static fire(eventName, data) {
		if (eventName == undefined) {
			throw new Error('An eventName must be specified');
		}
		let listeners = EventEmitter.#eventListener[eventName];
		if (listeners == undefined) {
			return;
		}
		for (let listener of listeners) {
			listener(data);
		}
	};
	
	/**
	 * Registers a listener function when an event with the given name is fired.
	 * @param eventName the name of the event
	 * @param listener the function that is called. It will get the optional data as argument 
	 * @returns a function that can be called to unregister the listener
	 * @throws Error if the eventName or listener is undefined
	 */
	static on(eventName, listener) {
		if (eventName == undefined) {
			throw new Error('An eventName must be specified');
		}
		if (listener == undefined) {
			throw new Error('A listener function must be specified');
		}
		let listeners = EventEmitter.#eventListener[eventName];
		if (listeners == undefined) {
			listeners = [];
			EventEmitter.#eventListener[eventName] = listeners;
		}
		if (listeners.indexOf(listener)<0) {
			listeners.push(listener);
		}
		return () => {
			EventEmitter.off(eventName, listener);
		};
	};
	
	/**
	 * Unregisters a listener function.
	 * @param eventName the name of the event
	 * @param listener the function that is to be unregistered. 
	 */
	static off(eventName, listener) {
		if (eventName == undefined) {
			throw new Error('An eventName must be specified');
		}
		if (listener == undefined) {
			throw new Error('A listener function must be specified');
		}
		let listeners = EventEmitter.#eventListener[eventName];
		if (listeners == undefined) {
			return;
		}
		let index = listeners.indexOf(listener);
		if (index >= 0) {
			listeners.splice(index, 1);
		}
	};
	    
	/*****************************************************************************************************************************************************************************************************************************/
	/*****************************************************************************************************************************************************************************************************************************/
	/** API END                                                                                                                                                                                                                 **/
	/*****************************************************************************************************************************************************************************************************************************/
	/*****************************************************************************************************************************************************************************************************************************/
	
};

/**
 * Legacy instance of the event emitter to be used in non module javascript files e.g. AngularJS services and controllers.
 */
window.legacyEventEmitter = {
	on: EventEmitter.on,
	off: EventEmitter.off,
	fire: EventEmitter.fire,
};
