import { KytosAttribute } from './kytos-attribute.js';

/**
 * KytosOn is a custom attribute handler that attaches an event listener to a DOM node.
 * The event type is determined by the attribute name (after 'k-on-').
 * When the event is triggered, it calls a callback function defined on the kytosElement.
 *
 * Usage:
 * <a href="" k-on-click="myClickHandler">Click me</a>
 * 
 * The myClickHandler function should be defined in the kytosElement class. It will get the event as parameter.
 * 
 * @extends KytosAttribute
 */
export class KytosOn extends KytosAttribute {

	/**
	 * Attaches an event listener to the specified node for the event type derived from the attribute name.
	 * The event handler is dynamically resolved from the kytosElement using the attribute value as the function name.
	 *
	 * @param {Object} kytosElement - The object containing potential event callback functions.
	 * @param {HTMLElement} node - The DOM node to which the event listener will be attached.
	 * @param {string} attrName - The attribute name, expected to be in the format 'on<EventName>'.
	 * @throws {Error} If no event callback function is found for the given attribute name or callback name.
	 */
	attached(kytosElement, node, attrName) {
		if (node._kytosOnCache == undefined) {
			node._kytosOnCache = {};
		}
		node._kytosOnCache[attrName] = this;
		if (this.eventListener != undefined) {
			node.removeEventListener(attrName.substring(5), this.eventListener);
			this.eventListener = undefined;
		}
		this.eventListener = (event) => {
			let eventCallback = node.getAttribute(attrName);
			if (eventCallback == undefined) {
				throw new Error(`No event callback function found for ${attrName}`);
			}
			if (kytosElement[eventCallback] == undefined || typeof kytosElement[eventCallback] != 'function') {
				throw new Error(`No event callback function found for ${eventCallback}`);
			}
			kytosElement[eventCallback](event);
		};
		node.addEventListener(attrName.substring(5), this.eventListener);
	};

	/**
	 * Detaches an event listener from the specified node.
	 *
	 * @param {HTMLElement} kytosElement - The custom element instance (not used in this method).
	 * @param {HTMLElement} node - The DOM node from which the event listener will be removed.
	 * @param {string} attrName - The attribute name, expected to start with 'on' (e.g., 'onClick').
	 */
	detached(kytosElement, node, attrName) {
		if (node._kytosOnCache == undefined) {
			return;
		}
		let instance = node._kytosOnCache[attrName];
		if (instance == undefined) {
			return;
		}
		if (instance.eventListener != undefined) {
			node.removeEventListener(attrName.substring(5), instance.eventListener);
			instance.eventListener = undefined;
		}
	};

};

KytosAttribute.define('k-on-click', KytosOn, false);
KytosAttribute.define('k-on-dblclick', KytosOn, false);
KytosAttribute.define('k-on-hover', KytosOn, false);
KytosAttribute.define('k-on-focus', KytosOn, false);
KytosAttribute.define('k-on-blur', KytosOn, false);
KytosAttribute.define('k-on-change', KytosOn, false);
KytosAttribute.define('k-on-value', KytosOn, false);
KytosAttribute.define('k-on-mousenter', KytosOn, false);
KytosAttribute.define('k-on-mouseleave', KytosOn, false);
KytosAttribute.define('k-on-keydown', KytosOn, false);
KytosAttribute.define('k-on-keyup', KytosOn, false);
KytosAttribute.define('k-on-keypress', KytosOn, false);
KytosAttribute.define('k-on-touchstart', KytosOn, false);
KytosAttribute.define('k-on-touchend', KytosOn, false);
KytosAttribute.define('k-on-touchmove', KytosOn, false);
KytosAttribute.define('k-on-touchcancel', KytosOn, false);
KytosAttribute.define('k-on-input', KytosOn, false);
KytosAttribute.define('k-on-submit', KytosOn, false);
KytosAttribute.define('k-on-scroll', KytosOn, false);
KytosAttribute.define('k-on-select', KytosOn, false);
KytosAttribute.define('k-on-contextmenu', KytosOn, false);
KytosAttribute.define('k-on-focusin', KytosOn, false);
KytosAttribute.define('k-on-focusout', KytosOn, false);
