/**
 * Base implementation of a custom Kytos attribute handling attributes at sub elements in KytosElements
 * 
 * <h2>Usage</h2>
 * <h3>KytosAttribute</h3>
 * <p>Create a script with the name my-test.js:</p>
 * <pre>
 * import {KytosAttribute} from './kytos-attribute.js';
 * 
 * export class MyAttribute extends KytosAttribute {
 * 
 *     attached(kytosElement, node, attrName) {
 *         node.textContent = node.getAttribute(attrName);
 *     };
 * 
 * }
 * 
 * KytosAttribute.define('my-test', MyTest);
 * </pre>
 * 
 * <h3>Referencing</h3>
 * <p>In your template in another KytosElement:</p>
 * <pre>
 * import {MyTest} from './my-test.js';
 * ...
 * <div my-test="Helo World"></div> 
 * ...
 * </pre>
 */
export class KytosAttribute {

	/*****************************************************************************************************************************************************************************************************************************/
	/*****************************************************************************************************************************************************************************************************************************/
	/** API methods and members                                                                                                                                                                                                 **/
	/*****************************************************************************************************************************************************************************************************************************/
	/*****************************************************************************************************************************************************************************************************************************/

	/** 
	 * Registers a KytosAttribute class at the given attribute name
	 * @param attrName the name of the attribute the class is registered for
	 * @param clazz the class, extending KytosAttribute
     * @param throwIfDuplicate true (default), if this method should throw errors on duplicet attribute registration 
	*/
	static define(attrName, clazz, throwIfDuplicate=true ) {
		if (this.#attributDefinitions[attrName] != undefined) {
            if ( throwIfDuplicate === true ) {
    			throw new Error('An KytosAttribute with the name "${attrName}" is already defined');                
            }
            return;
		}
		this.#attributDefinitions.set(attrName, clazz);
	};
	
	/**
	 * Returns whether this attribute needs to tell tha caller that all inner nodes should not be bound anymore, e.g. because they are only a template for repeating.
	 * @return whether this attribute needs to tell tha caller that all inner nodes should not be bound anymore.
	 */
	static skipInnerNodes() {
		return false;
	}
	
	/**
	 * Called as soon as the given kytosElement is attached to the DOM. Will called again when after the kytosElement was detachted and attached again.
	 * @param kytosElement the parent KytosElement instance that has a template with this attribute attached
	 * @param node the specific node that has this attribute
	 * @param attrName the name of the attribute
	 */
	attached(kytosElement, node, attrName) {
		throw new Error('attached(kytosElement, node, attrName) function must be implemented');
	};
	
	/**
	 * Called as soon as the given kytosElement is detached from the DOM. Will called again when after the kytosElement was attached and detachted again.
	 * @param kytosElement the parent KytosElement instance that has a template with this attribute attached
	 * @param node the specific node that has this attribute
	 * @param attrName the name of the attribute
	 */
	detached(kytosElement, node, attrName) {
		// Not implemented
	};
	    
	/*****************************************************************************************************************************************************************************************************************************/
	/*****************************************************************************************************************************************************************************************************************************/
	/** API END                                                                                                                                                                                                                 **/
	/*****************************************************************************************************************************************************************************************************************************/
	/*****************************************************************************************************************************************************************************************************************************/
	
	static #attributDefinitions = new Map();
	
	/**
	 * Determines whether the specified attribute should skip inner nodes.
	 *
	 * @private
	 * @param {string} attrName - The name of the attribute to check.
	 * @returns {boolean} True if the attribute should skip inner nodes; otherwise, false.
	 * @throws {Error} If no attribute name is specified.
	 */
	static _skipsInnerElements(attrName) {
		if (attrName == undefined) {
			throw new Error('An attribute name must be specified');
		}
		let clazz = this.#attributDefinitions.get(attrName);
		return clazz.skipInnerNodes();
	};
	
	/**
	 * Returns whether a KytosAttribute is registered under the given attribute name
	 * @param attrName the name of the attribute to be checked
	 * @return whether a KytosAttribute is registered for the given attribute name
	 */
	static _hasAttributeDefined(attrName) {
		if (attrName == undefined) {
			throw new Error('An attribute name must be specified');
		}
		return this.#attributDefinitions.has(attrName);
	};
	
	/**
	 * Internal API method!
	 * Called when the attribute is bound to a node and should be executed.
	 * It will create an instance of the KytosAttribute and call the initialization
	 * @param attrName the name of the attribute
	 * @param kytosElement the parent KytosElement instance that has a template with this attribute attached
     * @param node the specific node that has this attribute
	 */
	static _bindAttribute(attrName, kytosElement, node) {
		if (attrName == undefined) {
			throw new Error('An attribute name must be specified');
		}
		let clazz = this.#attributDefinitions.get(attrName);
		let attrInstance = new clazz(kytosElement, node);
		attrInstance.#init(kytosElement, node, attrName);
	};
	
	/**
	 * Internal API method!
	 * Called when the attribute is unbound from a node and should be executed.
	 * @param attrName the name of the attribute
	 * @param kytosElement the parent KytosElement instance that has a template with this attribute attached
     * @param node the specific node that has this attribute
	 */
	static _unbindAttribute(attrName, kytosElement, node) {
		if (attrName == undefined) {
			throw new Error('An attribute name must be specified');
		}
		let clazz = this.#attributDefinitions.get(attrName);
		let attrInstance = new clazz(kytosElement, node);
		attrInstance.detached(kytosElement, node, attrName);
	};
	
	/**
	 * Initialize the attribute and checks the given parameters. The attached method is called afterwards.
	 * @param kytosElement the parent KytosElement instance that has a template with this attribute attached
     * @param node the specific node that has this attribute
	 * @param attrName the name of the attribute
	 */
	#init(kytosElement, node, attrName) {
		if (kytosElement == undefined) {
			throw new Error('kytosElement must be specified');
		}
		if (node == undefined) {
			throw new Error('node must be specified');
		}
		this.attached(kytosElement, node, attrName);
	};
	
};
