#!/usr/bin/env python3
import importlib.util
import subprocess
import sys
import argparse

# List of required libraries
REQUIRED_LIBRARIES = [
    ("torch", "PyTorch"),
    ("torchvision", "PyTorch Vision"),
    ("transformers", "Hugging Face Transformers"),
    ("Pillow", "Pillow (PIL)"),
    ("numpy", "NumPy"),
    ("timm", "Timm"),
]

def check_library(library_name: str, display_name: str) -> bool:
    """
    Checks if a library is installed.
    """
    try:
        result = None
        if library_name == "Pillow":
            result = importlib.util.find_spec("PIL")
        else:
            result = importlib.util.find_spec(library_name)
        result = False if result is None else True
        return result
    except ImportError:
        # print(f"Library {display_name} needs to be installed.")
        return False

def install_library(library_name: str, display_name: str) -> bool:
    """
    Attempts to install a library using pip.
    """
    print(f"Attempting to install {display_name}...")
    try:
        result = subprocess.run(
            [sys.executable, "-m", "pip", "install", library_name],
            capture_output=True,
            text=True,
            check=True
        )
        print(f"{display_name} was successfully installed.")
        return True
    except subprocess.CalledProcessError as e:
        print(f"Error installing {display_name}: {e.stderr.strip()}")
        return False

def main(check_only: bool) -> int:
    """
    Main function: Checks libraries and optionally installs missing ones.
    """
    all_installed = True

    for lib_name, display_name in REQUIRED_LIBRARIES:
        if not check_library(lib_name, display_name):
            all_installed = False
            if check_only:
                print(f"Library {display_name} needs to be installed.")
                continue
            if not install_library(lib_name, display_name):
                print(f"Installation of {display_name} failed. Please install manually.")
                continue
            # Re-check after installation
            if not check_library(lib_name, display_name):
                print(f"Error: {display_name} could not be loaded after installation.")
                all_installed = False
            else:
                all_installed = True

    if all_installed:
        print("All required libraries are present.")
        return 0
    else:
        if not check_only:
            print("Some libraries need to be installed.")
        return 1
            

if __name__ == "__main__":
    parser = argparse.ArgumentParser(description="Check or install required libraries for Microsoft Table Transformer.")
    parser.add_argument(
        "--check-only",
        action="store_true",
        help="Only check if libraries are installed, do not attempt to install missing ones."
    )
    args = parser.parse_args()

    sys.exit(main(args.check_only))